--[[---------------------------------------------------------------------------
	Chocolatier Two Simulator: Shops
	Copyright (c) 2006-2007 Big Splash Games, LLC. All Rights Reserved.
--]]---------------------------------------------------------------------------

require("sim/building.lua")

-- Shop class definition, derived from LBuilding
LShop = LBuilding:subclass
{
	__tostring = function(t) return "{Shop:" .. tostring(t.name) .. "}" end,
	_Shops = {},
	
	ambient = "sfx/chocolate_shop_loop.ogg",
}

-------------------------------------------------------------------------------
-- Functions for data description

function Shop(t) return LShop:new(t) end

-------------------------------------------------------------------------------
-- "constructor"

function LShop:new(t)
	t = LBuilding.new(self, t)
	table.insert(self._Shops, t)
	return t
end

-------------------------------------------------------------------------------
-- Convenient data access

function LShop:AllShops()
	return bsgArrayIterator(self._Shops)
end

-------------------------------------------------------------------------------
-- Reset

function LShop:ResetOne()
	LBuilding.ResetOne(self)
	-- TODO: Reset configuration info
end

function LShop:SetPort(p)
	LBuilding.SetPort(self, p)
	p.hasShop = true
end

-------------------------------------------------------------------------------
-- Ownership

function LShop:MarkOwned()
	gSim.shopOwned = true
	LBuilding.MarkOwned(self)
	if self.port then
		self.port.shopOwned = true
	end
end

-------------------------------------------------------------------------------
-- Building interaction

function LShop:SelectCharacterFreePlay() return self.character[1] end

function LShop:OnActivate()
	gActiveBuilding = self
	self:SetAmbient()
	local char,quest = self:SelectCharacter()
	
	local items = {}
	for i in LItem:OwnedProducts() do
		table.insert(items, i)
	end
	
	if table.getn(items) > 0 then
		DisplayDialog { "ui/shopui.lua", shop=self, char=char, items=items }
	elseif not quest then
		local k = GetString("shop_noitems")
		DisplayDialog { "ui/chartalk.lua", char=char, body="#"..k, ok="ok" }
	end
		self:ExitBuilding()
end

-------------------------------------------------------------------------------
-- Haggling

function LShop:ComputeHaggle(c)
	-- The "reasonableness" of a current price is a measure of where the
	-- actual price lies between the item low and high prices
	local count = 0
	local R = 0
	for item in LItem:OwnedProducts() do
		-- HIGHER prices are better
		local target = (item.high - item.price) / (item.high - item.low)
		R = R + target
		count = count + 1
	end
	-- The overall "reasonableness" of prices is the average reasonableness
	R = 100 * R / count
	
	-- Factor in the "haggle factor" for this character -- just a multiplier for R
	-- Lower haggleFactor means lower R, means less likely to haggle
	if c.haggleFactor then R = R * c.haggleFactor end

	-- Adjust according to character mood -- 10% bump in either direction
	if c:IsAngry() then R = R * .9
	elseif c:IsHappy() then R = R * 1.1
	end
	
	-- Now on a scale of 1..100, the lower value of R, the better the prices...
	-- If we roll lower than R, we can get better prices -- so as prices get
	-- better, our chances of rolling even lower are slim
	local H = bsgRandom(1,100)
	
	-- Roll is 20 pts under, prices drop
	-- 30 pts or more over, prices rise
	-- SO... if actual price is half way between low and high, R=50
	--  R=100: 80% drop, 20% stay the same
	--  R=50: 30% drop, 50% stay the same, 20% go up
	--  R=30: 10% dop, 50% stay the same, 40% go up
	--  R=0: 0% drop, 30% stay the same, 70% go up
	
	-- Force a successful haggle based on quest variable setting
	if LQuest:Variable("HaggleBonus") > 0 then
		H = R - 21
		LQuest._Variables.HaggleBonus = LQuest._Variables.HaggleBonus - 1
	end

	local response = "none"
	if H < R-20 then
		-- Prices rise toward maximum
		response = "up"
		for item in LItem:OwnedProducts() do
			local newPrice = bsgFloor((item.price + item.high) / 2 + .5)
			item.price = newPrice
		end
		-- "Good" sound
		PlaySound("haggle_price_down")
	elseif H > R+30 then
		-- Prices lower to minimum, character becomes angry
		response = "down"
		for item in LItem:OwnedProducts() do
			item.price = item.low
		end
		c:SetMood(LCharacter.kAngry - 10)
		-- "Bad" sound
		PlaySound("haggle_price_up")
	end
	
	return response
end
